// File created: 2007-10-23 14:11:11

package ope.adventure.parsers;

import java.util.InputMismatchException;

import ope.adventure.util.BooleanExpression;
import ope.adventure.util.Utils;

public final class BooleanExpressionParser {
	private final String str;
	private       int    pos = 0;

	public BooleanExpressionParser(final String s) {
		str = s;
	}

	public int getEnd() {
		return pos;
	}

	// wholething    :: "(" bool-expr ")"
	//                | "(default)"
	// bool-expr     :: spell-name
	//                | operator-expr
	// operator-expr :: and-expr
	//                | or-expr
	//                | not-expr
	// &-expr        :: "&" sub-expr sub-expr
	// |-expr        :: "|" sub-expr sub-expr
	// !-expr        :: "!" sub-expr
	// sub-expr      :: bool-expr
	//                | "(" operator-expr ")"
	public BooleanExpression parse() {
		BooleanExpression root;

		expect('(');
		if (accept("default"))
			root = new BooleanExpression(BooleanExpression.Symbol.DEFAULT);
		else
			root = booleanExpression();
		expect(')');

		return root;
	}

	private BooleanExpression booleanExpression() {
		if (accept('"')) {
			int i = pos;
			while (str.charAt(i) != '"')
				++i;

			final BooleanExpression
				s = new BooleanExpression(Utils.toLower(str.substring(pos, i)));

			pos = i+1;

			return s;
		} else {
			final BooleanExpression oe = operatorExpression();

			if (oe == null)
				throw new BadBooleanExpressionException(
					"expected operator or spell, got " +
					(pos < str.length() ? str.charAt(pos) : "<EOF>"));

			return oe;
		}
	}

	private BooleanExpression operatorExpression() {
		if (accept('&'))
			return new BooleanExpression(
				BooleanExpression.Symbol.AND,
				subExpression(),
				subExpression());

		else if (accept('|'))
			return new BooleanExpression(
				BooleanExpression.Symbol.OR,
				subExpression(),
				subExpression());

		else if (accept('!'))
			return new BooleanExpression(
				BooleanExpression.Symbol.NOT,
				subExpression());

		else
			return null;
	}

	private BooleanExpression subExpression() {
		if (accept('(')) {
			final BooleanExpression oe = operatorExpression();

			if (oe == null)
				throw new BadBooleanExpressionException(
					"expected operator expression in brackets of subexpression");

			expect(')');

			return oe;
		} else
			return booleanExpression();
	}

	private boolean accept(final char c) {
		pos = Utils.skipWhiteSpace(pos, str);

		if (pos < str.length() && str.charAt(pos) == c) {
			++pos;
			return true;
		} else
			return false;
	}
	private boolean accept(final String s) {
		pos = Utils.skipWhiteSpace(pos, str);

		for (int i = 0, j = pos; i < s.length() && j < str.length(); ++i, ++j)
			if (str.charAt(j) != s.charAt(i))
				return false;

		pos += s.length();
		return pos < str.length();
	}

	private void expect(final char c) {
		if (!accept(c))
			throw new BadBooleanExpressionException(
				"expected '" +c+ "', got " +
				(pos < str.length() ? "'"+str.charAt(pos)+"'" : "<EOF>"));
	}
}

final class BadBooleanExpressionException extends InputMismatchException {
	public BadBooleanExpressionException(final String msg) {
		super(msg);
	}
}
