// File created: 2007-10-09 20:33:27

package ope.adventure;

import java.io.ByteArrayOutputStream;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.TreeSet;

import ope.adventure.actions.*;
import ope.adventure.book.Book;
import ope.adventure.book.BookSection;
import ope.adventure.book.ID;
import ope.adventure.util.Utils;

public abstract class Action {
	// Actions use the print* methods to write their output to out, which is
	// then stored in output. output is read externally through the getOutput()
	// method and thence passed to the UI.
	private static final ByteArrayOutputStream output;
	private static final PrintWriter           out;

	// Solely for the purpose of reassigning stderr in ReloadAction
	protected static final PrintStream replaceStderr;

	static {
		output        = new ByteArrayOutputStream();
		out           = new PrintWriter(output);
		replaceStderr = new PrintStream(output);
	}

	protected static final void printf(final String s, final Object... args) {
		out.printf(null, s, args);
	}
	protected static final void println() {
		out.println();
	}
	protected static final void println(final Object s) {
		out.println(s);
	}
	protected static final void printfln(final String s, final Object... args) {
		out.printf(null, s, args).println();
	}
	protected static final void printBook(final Book b) {
		printfln("%n%s", b.getFullDescription());
	}
	protected static final void printSection(final BookSection s) {
		println(s.getContents());
	}
	protected static final void printBookId(final Book b) {
		printfln("%8s. %s", ID.nextBookId(), b.getShortTitle());
	}
	protected static final void printSectionId(final String snippet) {
		printfln("%8s. %s", ID.nextSectionId(), snippet);
	}

	public static final String getOutput() {
		out.flush();
		final String s = output.toString();
		output.reset();
		assert output.size() == 0;
		return s;
	}

	public static final void haveNoData() {
		println(
			"Data isn't loaded, can't execute command!\n" +
			"Try the 'reload' command.");
	}

	private static final Map<String, Action>
		actions = new HashMap<String, Action>();

	private static void add(final Action act, final String... aliases) {
		for (final String alias : aliases)
			actions.put(alias, act);
	}

	static {
		add(new LookAction(),     "look", "examine");
		add(new MovementAction(), "go");
		add(new WaitAction(),     "wait");

		add(new ReadAction(),    "read");
		add(new BrowseAction(),  "browse");
		add(new SearchAction(),  "search", "find");
		add(new CloseAction(),   "close");
		add(new RecentAction(),  "recent");
		add(new ReadingAction(), "reading");

		add(new ReloadAction(), "reload");

		add(new MagicAction(),    "cast");
		add(new MemorizeAction(), "memorize", "alias");

		add(new RingAction(), "ring", "use");

		// 80 - "%8s - ".length = 69
		for (final Action a : actions.values())
			assert a.getShortHelp().length() <= 69;
	}

	private static final Pattern
		HELP_REQUEST = Pattern.compile("^(?:he?lp|\\?+)[!?]*$"),
		QUIT_REQUEST = Pattern.compile("^(?:exit|quit)$");

	private static final String HELP =
		"Welcome to ASiP, copyleft Matti Niemenmaa, 2007." +
		"\n\n" +
		"A basic sequence for beginners (the help topics can be skipped but " +
		"you probably\nwon't understand what's going on if you do):\n" +
		"    look at assignment\n" +
		"    go to library\n" +
		"    help search\n" +
		"    help browse\n" +
		"    help read\n" +
		"    help recent\n" +
		"    help reading\n" +
		"    help close\n" +
		"    search for <whatever it is you're looking for>\n" +
		"    read <a book identifier output by search>\n" +
		"    browse <book identifier>\n" +
		"    browse <book identifier> for <a keyword output by browse>\n" +
		"    read <a section identifier output by browse>\n" +
		"    [cast <a spell found in the book>]\n" +
		"\n" +
		"Repeat that until you figure you've got what it takes to summon and " +
		"control an\nentity. Then, 'go to laboratory' and cast the required " +
		"spells. If you're\nsatisfied with the results, 'go to lobby' and " +
		"'ring bell'. If not, get rid of\nwhatever you summoned and try again." +
		"\n\n" +
		"Of course, you can rush into the job, casting the first summoning " +
		"spell you find\nin the books. However, such an attempt is likely to " +
		"cost you your life, or\nworse." +
		"\n\n" +
		"For help on the config file, use 'help config'." +
		"\n\n" +
		"Full list of verbs (use 'help <verb>' to get detailed help):\n";

	public static final boolean doableWithoutBooks(final String command) {
		final Action act = actions.get(getHead(command));

		return act == null || act.doableWithoutBooks();
	}
	public static final boolean doableWithoutSpells(final String command) {
		final Action act = actions.get(getHead(command));

		return act == null || act.doableWithoutSpells();
	}

	// Return value indicates the new time after the action has been performed.
	public static final int perform(
		final String        command,
		final Configuration config,
		final Player        player
	) {
		int time = player.getTime();

		final String trimmed = command.trim();
		if (trimmed.length() == 0)
			return time;

		final String[] words = splitCommand(trimmed);
		final String head = words[0];

		if (HELP_REQUEST.matcher(head).matches()) {
			if (words.length > 1) {
				if (words[1].equals("config")) {
					println(Configuration.getHelp());
					return time;
				}

				final Action act = actions.get(words[1]);
				if (act != null) {
					printfln(
						"%s - %s%n%n%s",
						words[1], act.getShortHelp(),
						act.getLongHelp());
					return time;
				}
			}

			println(HELP);

			for (final String verb : new TreeSet<String>(actions.keySet()))
				  printfln("%8s - %s", verb, actions.get(verb).getShortHelp());

			return time;
		}

		if (QUIT_REQUEST.matcher(head).matches()) {
			println("Quitting...");
			player.quit();
			return time;
		}

		final Action act = actions.get(head);
		if (act == null) {
			printfln("'%s' isn't a known command.", head);
			return time;
		}

		if (act.stopReading())
			time += stopReading(player);

		// FIXME kludge, Spells need to know the time when casting began
		Spell.setTime(time);

		time += act.perform(
			Arrays.asList(words).subList(1, words.length),
			config, player);

		updateRooms(player, time);

		return time;
	}

	private static void updateRooms(final Player player, final int time) {

		for (final Room room : Room.getRooms()) {

			final Entity e = room.getEntity();
			if (e != null) {
				if (e.isLeaving())
					room.removeEntity();
				else {
					final Effect motion = e.beginMotion(
						player.getTime() + 1, time, room.getEffectSet());

					if (motion != null) {
						room.addEffect(motion);
						println();
						if (player.getRoom() == room)
							// see HACK in Entity.java: not expiring, entering
							println(motion.getExpirationMessage());
						else
							println("You sense a disturbance nearby.");
					}
				}
			}

			while (room.hasExpiringEffects(time)) {

				final Effect eff       = room.pollEffect();
				final String expireMsg = eff.getExpirationMessage();

				// spell effects don't have results
				if (eff.getResults() == null) {
					if (expireMsg != null && room == player.getRoom()) {
						println();
						println(expireMsg);
					}

					continue;
				}

				// if we get this far, it's an entity motion

				assert eff.getName()        == null;
				assert eff.getDescription() == null;
				assert room.getEntity()     != null;

				final Result
					res = Result.getResult(eff.getResults(), room.getEffectSet());

				if (res == null)
					throw new IllegalStateException(
						"Action :: no result for end of entity motion, " +
						"lack of a default should have been caught by parser");

				boolean showMessage = player.getRoom() == room;
				if (res.isSuccessful()) {
					final Result.Consequences consequences = res.getConsequences();

					consequences.apply(player, time, room);

					if (consequences.willLeave())
						room.removeEntity();

					showMessage |= consequences.willAlwaysQuit(); 
				}

				if (showMessage) {
					println();
					println(res.getMessage());
				}

				room.getEntity().endMotion(time);
			}
		}
	}

	private static String[] splitCommand(final String command) {
		return command.split("\\s+");
	}
	private static String getHead(final String command) {
		return command.split("\\s+", 2)[0];
	}

	protected static final void ignoringExcess(
		final List<String> words, final int lim
	) {
		if (words.size() > lim)
			printfln(
				"(Ignoring excess '%s'...)",
				Utils.join(words.subList(lim, words.size()), " "));
	}

	protected static final int stopReading(final Player player) {
		int time = 0;
		while (player.isReading())
			time += closeBook(player.popBook());
		return time;
	}
	protected static final int closeBook(final Book book) {
		printfln(
			"You close '%s' and place it back where it belongs.",
			book.getShortTitle());
		return 1;
	}

	public abstract int perform(List<String> w, Configuration c, Player p);

	public abstract boolean doableWithoutBooks();
	public abstract boolean doableWithoutSpells();
	public abstract boolean stopReading();
	
	public abstract String getShortHelp();
	public abstract String getLongHelp();
}
